import { useEffect, useState } from "react";
import { useLocation, useNavigate, useSearchParams } from "react-router-dom";
import Header from "@/components/Header";
import PointsDisplay from "@/components/PointsDisplay";
import ProductGridSalt from "@/components/ProductGridSalt";
import LoadingSkeleton from "@/components/LoadingSkeleton";
import { Language } from "@/components/LanguageSwitcher";
import { useTranslation } from "@/lib/translations";
import { ArrowLeft, Sparkles, Mail, Award } from "lucide-react";
import { Button } from "@/components/ui/button";

// ========================================
// 🔗 PUT YOUR PAYMENT LINK HERE:
// ========================================
const PAYMENT_LINK = "https://your-payment-link.com";
// ========================================

const Rewards = () => {
  const location = useLocation();
  const navigate = useNavigate();
  const [searchParams] = useSearchParams();
  const [isLoading, setIsLoading] = useState(true);
  const [userEmail, setUserEmail] = useState("");
  const [language, setLanguage] = useState<Language>("de");
  const t = useTranslation(language);
  const userPoints = 590;

  useEffect(() => {
    // Get email from navigation state, URL params, or localStorage
    const email = location.state?.email || searchParams.get('email') || localStorage.getItem('userEmail');
    const lang = searchParams.get('lang') || localStorage.getItem('userLanguage');
    
    if (!email) {
      navigate("/");
      return;
    }
    
    setUserEmail(email);
    localStorage.setItem('userEmail', email);
    
    // Set language if provided
    if (lang && ['de', 'fr', 'it', 'en'].includes(lang)) {
      setLanguage(lang as Language);
    }

    // Simulate loading
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, 2000);

    return () => clearTimeout(timer);
  }, [location.state, navigate, searchParams]);

  if (isLoading) {
    return (
      <>
        <Header language={language} onLanguageChange={setLanguage} t={t} />
        <LoadingSkeleton />
      </>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <Header language={language} onLanguageChange={setLanguage} t={t} />
      
      <main className="pt-24 pb-16">
        <div className="container mx-auto px-4">
          {/* Back button */}
          <Button
            variant="ghost"
            onClick={() => navigate("/")}
            className="mb-8 text-muted-foreground hover:text-foreground"
          >
            <ArrowLeft className="h-4 w-4 mr-2" />
            {t("back")}
          </Button>

          {/* User info card */}
          <div className="max-w-lg mx-auto mb-8 animate-fade-in">
            <div className="bg-card border border-border rounded-2xl p-6 flex items-center gap-4">
              <div className="w-12 h-12 rounded-full bg-accent/10 flex items-center justify-center">
                <Mail className="h-5 w-5 text-accent" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">{t("welcomeBack")}</p>
                <p className="font-semibold text-foreground">{userEmail}</p>
              </div>
              <div className="ml-auto flex items-center gap-2 bg-accent/10 text-accent px-3 py-1.5 rounded-full">
                <Award className="h-4 w-4" />
                <span className="text-sm font-semibold">{userPoints} pts</span>
              </div>
            </div>
            <div className="mt-3 px-4 py-2 bg-destructive/10 border border-destructive/20 rounded-lg text-center">
              <p className="text-sm text-destructive font-medium">
                ⏰ {t("pointsExpire")}
              </p>
            </div>
          </div>

          {/* Points Display */}
          <div className="max-w-lg mx-auto mb-16">
            <PointsDisplay points={userPoints} t={t} />
          </div>

          {/* Products Section */}
          <section>
            <div className="text-center mb-12">
              <div className="inline-flex items-center gap-2 text-accent mb-4">
                <Sparkles className="h-5 w-5" />
                <span className="text-sm font-semibold uppercase tracking-wider">
                  {t("pointsRewards")}
                </span>
              </div>
              <h2 className="text-3xl md:text-4xl font-extrabold text-foreground mb-4">
                {t("exclusiveDeals")}
              </h2>
              <p className="text-muted-foreground max-w-xl mx-auto">
                {t("exclusiveDealsDesc")}
              </p>
            </div>

            <ProductGridSalt paymentLink={PAYMENT_LINK} t={t} />
          </section>

          {/* Info banner */}
          <div className="mt-16 bg-accent/5 border border-accent/20 rounded-2xl p-8 text-center animate-fade-in">
            <h3 className="font-bold text-lg text-foreground mb-2">
              {t("howItWorks")}
            </h3>
            <p className="text-muted-foreground max-w-2xl mx-auto">
              {t("howItWorksDesc")}
            </p>
          </div>
        </div>
      </main>

      {/* Footer */}
      <footer className="border-t border-border py-8">
        <div className="container mx-auto px-4 text-center">
          <p className="text-sm text-muted-foreground">
            © {new Date().getFullYear()} Salt Rewards. {t("allRightsReserved")}
          </p>
        </div>
      </footer>
    </div>
  );
};

export default Rewards;
